<?php

namespace App\Services;

use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Http;
use Illuminate\Http\Client\RequestException;
use Psr\Log\LoggerInterface;

class WfcoClient
{
    protected string $base;
    protected string $user;
    protected string $pass;
    protected int $timeout;
    protected LoggerInterface $log;

    public function __construct(LoggerInterface $log)
    {
        $cfg = config('services.wfco');
        $this->base    = rtrim($cfg['host'], '/');
        $this->user    = $cfg['username'];
        $this->pass    = $cfg['password'];
        $this->timeout = $cfg['timeout'] ?? 20;
        $this->log     = $log;
    }

    protected function http()
    {
        return Http::baseUrl($this->base)
            ->acceptJson()
            ->asJson()
            ->timeout($this->timeout)
            ->withHeaders([
                'Content-Type' => 'application/json',
            ])->retry(3, 500, function ($exception, $request) {
                // retry on connect/read timeouts
                return $exception instanceof \Illuminate\Http\Client\ConnectionException;
            })
            ->withOptions([
                // Timeouts
                'connect_timeout' => 5,     // fail fast if handshake can’t start
                'timeout'         => 25,    // whole request budget
                // SSL CA (fix your earlier cURL 60)
                // 'verify'          => env('WFCO_CA_BUNDLE', true),
                // // Many servers prefer HTTP/1.1; also avoids some HTTP/2+TLS quirks
                // 'version'         => CURL_HTTP_VERSION_1_1,
                // // Force IPv4 if IPv6 is flaky
                // 'force_ip_resolve' => 'v4',
                // // Keepalives help if making many calls
                // 'curl' => [
                //     CURLOPT_TCP_KEEPALIVE => 1,
                //     CURLOPT_TCP_KEEPIDLE  => 30,
                //     CURLOPT_TCP_KEEPINTVL => 15,
                // ],
            ]);
    }

    /**
     * Get OAuth token using Basic Auth.
     * Body must include "approveCode" per Postman file.
     */
    public function oauth(?string $approveCode = null): string
    {
        $approveCode = $approveCode ?? config('services.wfco.approve_code');

        $res = $this->http()
            ->withBasicAuth($this->user, $this->pass)
            ->post('/external/v1/merchant/oauth', [
                'approveCode' => $approveCode,
            ]);

        if ($res->failed()) {
            $this->log->error('WFCO OAuth failure', ['status' => $res->status(), 'body' => $res->json()]);
            $res->throw(); // throws RequestException
        }

        $data = $res->json();

        // Try common token field names
        $token = $data['token']
            ?? $data['accessToken']
            ?? $data['access_token']
            ?? null;

        if (!$token) {
            throw new \RuntimeException('WFCO OAuth: token not found in response');
        }

        // cache ~45 minutes unless API returns exp
        $ttl = now()->addMinutes(45);

        // If API returns exp (epoch seconds) use that
        if (isset($data['exp']) && is_numeric($data['exp'])) {
            $exp = \Carbon\Carbon::createFromTimestamp($data['exp']);
            // subtract a small skew to be safe
            $ttl = $exp->subMinutes(2);
        }

        Cache::put($this->cacheKey(), $token, $ttl);
        return $token;
    }

    protected function cacheKey(): string
    {
        return 'wfco.oauth.token';
    }

    protected function token(): string
    {
        return Cache::remember($this->cacheKey(), 2700, function () {
            return $this->oauth(); // will throw on failure
        });
    }

    protected function authed()
    {
        return $this->http()->withToken($this->token());
    }

    /** Initiate PO */
    public function initiatePo(array $payload): array
    {
        // expects keys: approval_code, invoice_number, otp_required, item_Info[]
        $res = $this->authed()->post('/external/v1/merchant/po/initiate', $payload);
        return $this->handle($res, 'initiatePo');
    }

    /** Submit OTP */
    public function otp(array $payload): array
    {
        // expects: approval_code, invoice_number, otp_Code
        $res = $this->authed()->post('/external/v1/merchant/po/otp', $payload);
        return $this->handle($res, 'otp');
    }

    /** Inquire PO */
    public function inquirePo(array $payload): array
    {
        // expects: approval_code, invoice_number, po_Number
        $res = $this->authed()->post('/external/v1/merchant/po/inquire', $payload);
        return $this->handle($res, 'inquirePo');
    }

    /** Delivery confirmation */
    public function delivery(array $payload): array
    {
        // expects: approval_code, invoice_number, delivery_number, id_number
        $res = $this->authed()->post('/external/v1/merchant/delivery', $payload);
        return $this->handle($res, 'delivery');
    }

    /** Cancel PO */
    public function cancelPo(array $payload): array
    {
        // expects: po_Number, invoice_number, amount, enquiryFlag
        $res = $this->authed()->post('/external/v1/merchant/po/cancel', $payload);
        return $this->handle($res, 'cancelPo');
    }

    /** Cancel Contract */
    public function cancelContract(array $payload): array
    {
        // expects: po_Number, invoice_number, amount, enquiryFlag
        $res = $this->authed()->post('/external/v1/merchant/contract/cancel', $payload);
        return $this->handle($res, 'cancelContract');
    }

    protected function handle($res, string $op): array
    {
        if ($res->unauthorized()) {
            // token might be expired or invalid: refresh once and retry
            Cache::forget($this->cacheKey());
            $res = $this->authed()->send($res->transferStats->getRequest()->getMethod(), $res->transferStats->getRequest()->getUri(), [
                'json' => json_decode($res->transferStats->getRequest()->getBody(), true)
            ]);
        }

        if ($res->failed()) {
            $this->log->error("WFCO {$op} failure", ['status' => $res->status(), 'body' => $res->json()]);
            $res->throw();
        }

        return $res->json() ?? [];
    }
}
